unit Errortbl;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, DB, DBTables, DBITypes, DBIProcs, DBIErrs;

type

  TErrorTable = class(TTable)
  private
    FPrevOnException: TExceptionEvent;
    FOnKeyViolation: TExceptionEvent;
    FOnMinCheckFail: TExceptionEvent;
    FOnMaxCheckFail: TExceptionEvent;
    FOnFldRequired: TExceptionEvent;
    FOnMasterMissing: TExceptionEvent;
    FOnLookupTblFail: TExceptionEvent;
    FOnRecLockFail: TExceptionEvent;
    FOnRecUnLockFail: TExceptionEvent;
	 FOnFileIsBusy: TExceptionEvent;
    FOnFileIsLocked: TExceptionEvent;
    FOnDirIsLocked: TExceptionEvent;
	 FOnMultipleNetFiles: TExceptionEvent;
	 FOnDetailRecordsExist: TExceptionEvent;
    FOnOtherErrors: TExceptionEvent;
    FErrorToken: string;
    FErrorTableName: string;
    FErrorFieldName: string;
    FErrorFieldDispName: string;
    FErrorFieldMinValue: string;
    FErrorFieldMaxValue: string;
    FErrorLookupTableName: string;
    FErrorImageRow: string;
    FErrorUserName: string;
    FErrorFileName: string;
    FErrorIndexName: string;
    FErrorDirName: string;
    FErrorKeyName: string;
    FErrorAlias: string;
    FErrorDriveName: string;
    FErrorNativeCode: string;
    FErrorNativeMsg: string;
    FErrorLineNumber: string;
    FErrorCapability: string;
	 FTableDescription: string;
	 FShouldAbort: Boolean;
	 function ErrorDlg(DisplayMsg: string; TitleMsg: string): Integer;
    procedure OnError(Sender: TObject;E: Exception);
    procedure AssignProps;
	 function GetRecordNumber: Longint;
	 procedure KeyViolationMsg;
	 procedure MinCheckFailMsg;
	 procedure MaxCheckFailMsg;
	 procedure FldRequiredMsg;
	 procedure MasterMissingMsg;
	 procedure LookupTblFailMsg;
	 procedure RecLockFailMsg;
	 procedure RecUnLockFailMsg;
	 procedure FileIsBusyMsg;
	 procedure FileIsLockedMsg;
	 procedure DirIsLockedMsg;
	 procedure MultipleNetFilesMsg;
	 procedure DetailRecordsExistMsg;
	 procedure OtherErrorsMsg(E: Exception);
  protected
    procedure DoBeforePost; override;
    procedure DoAfterPost; override;
    procedure DoBeforeEdit; override;
    procedure DoAfterEdit; override;
    procedure DoBeforeInsert; override;
    procedure DoAfterInsert; override;
    procedure DoBeforeDelete; override;
    procedure DoAfterDelete; override;
    procedure DoBeforeCancel; override;
    procedure DoAfterCancel; override;
    procedure DoBeforeOpen; override;
    procedure DoAfterOpen; override;
    procedure DoBeforeClose; override;
    procedure DoAfterClose; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property OnKeyViolation: TExceptionEvent read FOnKeyViolation write FOnKeyViolation;
    property OnMinCheckFail: TExceptionEvent read FOnMinCheckFail write FOnMinCheckFail;
    property OnMaxCheckFail: TExceptionEvent read FOnMaxCheckFail write FOnMaxCheckFail;
    property OnFldRequired: TExceptionEvent read FOnFldRequired write FOnFldRequired;
    property OnMasterMissing: TExceptionEvent read FOnMasterMissing write FOnMasterMissing;
    property OnLookupTblFail: TExceptionEvent read FOnLookupTblFail write FOnLookupTblFail;
    property OnRecLockFail: TExceptionEvent read FOnRecLockFail write FOnRecLockFail;
    property OnRecUnLockFail: TExceptionEvent read FOnRecUnlockFail write FOnRecUnlockFail;
    property OnFileIsBusy: TExceptionEvent read FOnFileIsBusy write FOnFileIsBusy;
    property OnFileIsLocked: TExceptionEvent read FOnFileIsLocked write FOnFileIsLocked;
    property OnDirIsLocked: TExceptionEvent read FOnDirIsLocked write FOnDirIsLocked;
    property OnMultipleNetFiles: TExceptionEvent read FOnMultipleNetFiles write FOnMultipleNetFiles;
    property OnDetailRecordsExist: TExceptionEvent read FOnDetailRecordsExist write FOnDetailRecordsExist;
    property OnOtherErrors: TExceptionEvent read FOnOtherErrors write FOnOtherErrors;
    property ErrorToken: string read FErrorToken;
    property ErrorTableName: string read FErrorTableName;
    property ErrorFieldName: string read FErrorFieldName;
    property ErrorFieldDispName: string read FErrorFieldDispName;
    property ErrorFieldMinValue: string read FErrorFieldMinValue;
    property ErrorFieldMaxValue: string read FErrorFieldMaxValue;
    property ErrorImageRow: string read FErrorImageRow;
    property ErrorUserName: string read FErrorUserName;
    property ErrorFileName: string read FErrorFileName;
    property ErrorIndexName: string read FErrorIndexName;
    property ErrorDirName: string read FErrorDirName;
    property ErrorKeyName: string read FErrorKeyName;
    property ErrorAlias: string read FErrorAlias;
    property ErrorDriveName: string read FErrorDriveName;
    property ErrorNativeCode: string read FErrorNativeCode;
    property ErrorNativeMsg: string read FErrorNativeMsg;
    property ErrorLineNumber: string read FErrorLineNumber;
    property ErrorCapability: string read FErrorCapability;
	 property RecordNumber: Longint read GetRecordNumber;
	 property TableDescription: string read FTableDescription write FTableDescription;
  end;

procedure Register;
function TableCallBack(ecbType: CBType; ClientData: Longint;
	                     var CBInfo: Pointer): CBRType; export;

implementation

procedure Register;
begin
  RegisterComponents('Data Access', [TErrorTable]);
end;

constructor TErrorTable.Create(AOwner: TComponent);
begin
	inherited Create(AOwner);
   FErrorToken:='';
   FErrorTableName:='';
   FErrorFieldName:='';
   FErrorFieldDispName:='';
   FErrorFieldMinValue:='';
   FErrorFieldMaxValue:='';
   FErrorLookupTableName:='';
   FErrorImageRow:='';
   FErrorUserName:='';
   FErrorFileName:='';
   FErrorIndexName:='';
   FErrorDirName:='';
   FErrorKeyName:='';
   FErrorAlias:='';
   FErrorDriveName:='';
   FErrorNativeCode:='';
   FErrorNativeMsg:='';
   FErrorLineNumber:='';
   FErrorCapability:='';
   FTableDescription:='';
   FShouldAbort:=False;
end;

procedure TErrorTable.DoBeforePost;
begin
   inherited DoBeforePost;
	FShouldAbort:=False;
   FPrevOnException:=Application.OnException;
   Application.OnException:=OnError;
end;

procedure TErrorTable.DoAfterPost;
begin
	DbiSaveChanges(Handle);
   Application.OnException:=FPrevOnException;
   inherited DoAfterPost;
end;

procedure TErrorTable.DoBeforeEdit;
begin
   inherited DoBeforeEdit;
	FShouldAbort:=False;
   FPrevOnException:=Application.OnException;
   Application.OnException:=OnError;
end;

procedure TErrorTable.DoAfterEdit;
begin
   Application.OnException:=FPrevOnException;
   inherited DoAfterEdit;
end;

procedure TErrorTable.DoBeforeInsert;
begin
   inherited DoBeforeInsert;
	FShouldAbort:=False;
   FPrevOnException:=Application.OnException;
   Application.OnException:=OnError;
end;

procedure TErrorTable.DoAfterInsert;
begin
   Application.OnException:=FPrevOnException;
   inherited DoAfterInsert;
end;

procedure TErrorTable.DoBeforeDelete;
begin
   inherited DoBeforeDelete;
	FShouldAbort:=False;
   FPrevOnException:=Application.OnException;
   Application.OnException:=OnError;
end;

procedure TErrorTable.DoAfterDelete;
begin
   Application.OnException:=FPrevOnException;
   inherited DoAfterDelete;
end;

procedure TErrorTable.DoBeforeCancel;
begin
   inherited DoBeforeCancel;
	FShouldAbort:=False;
   FPrevOnException:=Application.OnException;
   Application.OnException:=OnError;
end;

procedure TErrorTable.DoAfterCancel;
begin
   Application.OnException:=FPrevOnException;
   inherited DoAfterCancel;
end;

procedure TErrorTable.DoBeforeOpen;
begin
   inherited DoBeforeOpen;
	FShouldAbort:=True;
   FPrevOnException:=Application.OnException;
   Application.OnException:=OnError;
end;

procedure TErrorTable.DoAfterOpen;
begin
	Check(DbiRegisterCallBack(Handle,cbTableChanged,Longint(Self),0,nil,TableCallBack));
   Application.OnException:=FPrevOnException;
   inherited DoAfterOpen;
end;

procedure TErrorTable.DoBeforeClose;
begin
	Check(DbiRegisterCallBack(Handle,cbTableChanged,Longint(Handle),0,nil,nil));
   inherited DoBeforeClose;
	FShouldAbort:=False;
   FPrevOnException:=Application.OnException;
   Application.OnException:=OnError;
end;

procedure TErrorTable.DoAfterClose;
begin
   Application.OnException:=FPrevOnException;
   inherited DoAfterClose;
end;

procedure TErrorTable.OnError(Sender: TObject;E: Exception);
begin
  if (E is EDatabaseError) then
     begin
        if (E is EDBEngineError) then
           begin
           AssignProps;
           with E as EDBEngineError do
             case Errors[0].ErrorCode of
                DBIERR_KEYVIOL:
						if Assigned(FOnKeyViolation) then
							FOnKeyViolation(Self,E)
						else
							KeyViolationMsg;
                DBIERR_MINVALERR:
						if Assigned(FOnMinCheckFail) then
							FOnMinCheckFail(Self,E)
						else
							MinCheckFailMsg;
                DBIERR_MAXVALERR:
						if Assigned(FOnMaxCheckFail) then
                  	FOnMaxCheckFail(Self,E)
						else
							MaxCheckFailMsg;
                DBIERR_REQDERR:
						if Assigned(FOnFldRequired) then
							FOnFldRequired(Self,E)
						else
							FldRequiredMsg;
                DBIERR_FORIEGNKEYERR:
						if Assigned(FOnMasterMissing) then
							FOnMasterMissing(Self,E)
						else
							MasterMissingMsg;
                DBIERR_LOOKUPTABLEERR:
						if Assigned(FOnLookupTblFail) then
							FOnLookupTblFail(Self,E)
						else
							LookupTblFailMsg;
                DBIERR_LOCKED:
						if Assigned(FOnRecLockFail) then
							FOnRecLockFail(Self,E)
						else
							RecLockFailMsg;
                DBIERR_UNLOCKFAILED:
						if Assigned(FOnRecUnLockFail) then
							FOnRecUnLockFail(Self,E)
						else
							RecUnLockFailMsg;
                DBIERR_FILEBUSY:
						if Assigned(FOnFileIsBusy) then
							FOnFileIsBusy(Self,E)
						else
							FileIsBusyMsg;
                DBIERR_FILELOCKED:
						if Assigned(FOnFileIsLocked) then
							FOnFileIsLocked(Self,E)
						else
							FileIsLockedMsg;
                DBIERR_DIRLOCKED:
						if Assigned(FOnDirIsLocked) then
							FOnDirIsLocked(Self,E)
						else
							DirIsLockedMsg;
                DBIERR_DETAILRECORDSEXIST:
						if Assigned(FOnDetailRecordsExist)then
                  	FOnDetailRecordsExist(Self,E)
						else
							DetailRecordsExistMsg;
                DBIERR_NETMULTIPLE:
						if Assigned(FOnMultipleNetFiles) then
							FOnMultipleNetFiles(Self,E)
						else
							MultipleNetFilesMsg;
             else
	         	if Assigned(FOnOtherErrors) then
                	FOnOtherErrors(Self,E)
  					else
				  		OtherErrorsMsg(E);
             end
         end
	   else
         if Assigned(FOnOtherErrors) then
            FOnOtherErrors(Self,E)
         else
            OtherErrorsMsg(E);
      end
   else
   	if Assigned(FOnOtherErrors) then
         FOnOtherErrors(Self,E)
      else
         OtherErrorsMsg(E);
Application.OnException:=FPrevOnException;
if FShouldAbort then SysUtils.Abort;
end;

procedure TErrorTable.AssignProps;
var
   pContext: PChar;
   ErrorCode: DBIResult;
   FldCtr: Integer;
   FldPos: Integer;
   ValChkCtr: Integer;
   TblProps: CURProps;
   pValChkDesc: pVCHKDesc;
   MinInteger: Integer;
   MaxInteger: Integer;
   MinSmallInt: Smallint;
   MaxSmallInt: Smallint;
   MinWord: Word;
   MaxWord: Word;
   MinFloat: Double;
   MaxFloat: Double;
   MinCurrency: Double;
   MaxCurrency: Double;
   MinDateTime: TDateTime;
   MaxDateTime: TDateTime;
begin
   try
      GetMem(pContext,DBIMAXMSGLEN+1);
      GetMem(pValChkDesc,SizeOf(VCHKDesc));

      ErrorCode:=DbiGetErrorContext(ecTOKEN,pContext);
      FErrorToken:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecTABLENAME,pContext);
      FErrorTableName:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecFIELDNAME,pContext);
      FErrorFieldName:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecIMAGEROW,pContext);
      FErrorImageRow:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecUSERNAME,pContext);
      FErrorUserName:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecFILENAME,pContext);
      FErrorFileName:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecINDEXNAME,pContext);
      FErrorIndexName:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecDIRNAME,pContext);
      FErrorDirName:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecKEYNAME,pContext);
      FErrorKeyName:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecALIAS,pContext);
      FErrorAlias:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecDRIVENAME,pContext);
      FErrorDriveName:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecNATIVECODE,pContext);
      FErrorNativeCode:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecNATIVEMSG,pContext);
      FErrorNativeMsg:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecLINENUMBER,pContext);
      FErrorLineNumber:=StrPas(pContext);
      ErrorCode:=DbiGetErrorContext(ecCAPABILITY,pContext);
      FErrorCapability:=StrPas(pContext);

      FldCtr:=0;
      FldPos:=0;

		if FErrorFieldName <> '' then
			begin
         for FldCtr:=0 to (FieldCount-1) do
            begin
            if (Fields[FldCtr].FieldName=FErrorFieldName) then
               begin
               FErrorFieldDispName:=Fields[FldCtr].DisplayLabel;
               FldPos:=FldCtr+1;
               Fields[FldCtr].FocusControl;
               Break;
               end
            end
			end;

      if FldPos <> 0 then
			begin
	      if DbiGetCursorProps(Handle,TblProps)=DBIERR_NONE then
            begin
            for ValChkCtr:=1 to TblProps.iValChecks do
               if DbiGetVchkDesc(Handle,ValChkCtr,pValChkDesc)=DBIERR_NONE then
                  begin
                  if pValChkDesc^.iFldNum=FldPos then
                     begin
                     if Fields[FldPos-1] is TIntegerField then
                        begin
                        Move(pValChkDesc^.aMinVal,MinInteger,SizeOf(LongInt));
                        Move(pValChkDesc^.aMaxVal,MaxInteger,SizeOf(LongInt));
                        FErrorFieldMinValue:=IntToStr(MinInteger);
                        FErrorFieldMaxValue:=IntToStr(MaxInteger);
                        Break;
                        end;
                     if Fields[FldPos-1] is TSmallIntField then
                        begin
                        Move(pValChkDesc^.aMinVal,MinSmallInt,SizeOf(SmallInt));
                        Move(pValChkDesc^.aMaxVal,MaxSmallInt,SizeOf(SmallInt));
                        FErrorFieldMinValue:=IntToStr(MinSmallInt);
                        FErrorFieldMaxValue:=IntToStr(MaxSmallInt);
                        Break;
                        end;
                     if Fields[FldPos-1] is TWordField then
                        begin
                        Move(pValChkDesc^.aMinVal,MinWord,SizeOf(Word));
                        Move(pValChkDesc^.aMaxVal,MaxWord,SizeOf(Word));
                        FErrorFieldMinValue:=IntToStr(MinWord);
                        FErrorFieldMaxValue:=IntToStr(MaxWord);
                        Break;
                        end;
                     if Fields[FldPos-1] is TFloatField then
                        begin
                        Move(pValChkDesc^.aMinVal,MinFloat,SizeOf(Double));
                        Move(pValChkDesc^.aMaxVal,MaxFloat,SizeOf(Double));
                        FErrorFieldMinValue:=FloatToStr(MinFloat);
                        FErrorFieldMaxValue:=FloatToStr(MaxFloat);
                        Break;
                        end;
                     if Fields[FldPos-1] is TCurrencyField then
                        begin
                        Move(pValChkDesc^.aMinVal,MinCurrency,SizeOf(Double));
                        Move(pValChkDesc^.aMaxVal,MaxCurrency,SizeOf(Double));
                        FErrorFieldMinValue:=FloatToStrF(MinCurrency,ffCurrency,15,2);
                        FErrorFieldMaxValue:=FloatToStrF(MaxCurrency,ffCurrency,15,2);
                        Break;
                        end;
                     if Fields[FldPos-1] is TDateTimeField then
                        begin
                        Move(pValChkDesc^.aMinVal,MinDateTime,SizeOf(TDateTime));
                        Move(pValChkDesc^.aMaxVal,MaxDateTime,SizeOf(TDateTime));
                        FErrorFieldMinValue:=DateTimeToStr(MinDateTime);
                        FErrorFieldMaxValue:=DateTimeToStr(MaxDateTime);
                        Break;
                        end;
                     if Fields[FldPos-1] is TDateField then
                        begin
                        Move(pValChkDesc^.aMinVal,MinDateTime,SizeOf(TDateTime));
                        Move(pValChkDesc^.aMaxVal,MaxDateTime,SizeOf(TDateTime));
                        FErrorFieldMinValue:=DateToStr(MinDateTime);
                        FErrorFieldMaxValue:=DateToStr(MaxDateTime);
                        Break;
                        end;
                     if Fields[FldPos-1] is TTimeField then
                        begin
                        Move(pValChkDesc^.aMinVal,MinDateTime,SizeOf(TDateTime));
                        Move(pValChkDesc^.aMaxVal,MaxDateTime,SizeOf(TDateTime));
                        FErrorFieldMinValue:=TimeToStr(MinDateTime);
                        FErrorFieldMaxValue:=TimeToStr(MaxDateTime);
                        Break;
                        end;
                     end
                  end
            end
         else
            begin
            FErrorFieldMinValue:='';
            FErrorFieldMaxValue:='';
            end
         end;

   finally
		if pContext <> nil then FreeMem(pContext,DBIMAXMSGLEN+1);
      if pValChkDesc <> nil then FreeMem(pValChkDesc,SizeOf(VCHKDesc));
   end;
end;

function TErrorTable.GetRecordNumber: Longint;
var
	CursorProps: CurProps;
	RecordProps: RECProps;
begin
	Result:=0;
	if State=dsInactive then
      raise EDatabaseError.Create('Cannot perform this operation '+
                                  'on a closed dataset.');

   Check(DbiGetCursorProps(Handle,CursorProps));
	UpdateCursorPos;
	Check(DbiGetRecord(Handle,dbiNOLOCK,nil,@RecordProps));

   case CursorProps.iSeqNums of
		0: Result:=RecordProps.iPhyRecNum;
		1: Result:=RecordProps.iSeqNum;
		end;
end;

procedure TErrorTable.KeyViolationMsg;
begin
	ErrorDlg('This '+FTableDescription+' already exists, please re-enter.','Data Entry Error');
end;

procedure TErrorTable.MinCheckFailMsg;
begin
	ErrorDlg('The '+FErrorFieldDispName+' must be greater than or equal to '+
   		   FErrorFieldMinValue+', please re-enter.','Data Entry Error');
end;

procedure TErrorTable.MaxCheckFailMsg;
begin
	ErrorDlg('The '+FErrorFieldDispName+' must be less than or equal to '+
   		   FErrorFieldMaxValue+', please re-enter.','Data Entry Error');
end;

procedure TErrorTable.FldRequiredMsg;
begin
	ErrorDlg('The '+FErrorFieldDispName+' must be entered.','Data Entry Error');
end;

procedure TErrorTable.MasterMissingMsg;
begin
	ErrorDlg('The master record for this '+FTableDescription+' is missing.','Internal Data Error');
end;

procedure TErrorTable.LookupTblFailMsg;
begin
	ErrorDlg('The lookup for the '+FErrorFieldDispName+' has failed.','Internal Data Error');
end;

procedure TErrorTable.RecLockFailMsg;
begin
	ErrorDlg('This '+FTableDescription+' is currently being edited by '+
    			FErrorUserName+', please try again later.','Multi-User Problem');
end;

procedure TErrorTable.RecUnLockFailMsg;
begin
	ErrorDlg('This '+FTableDescription+' could not be unlocked.','Internal Data Error');
end;

procedure TErrorTable.FileIsBusyMsg;
begin
	ErrorDlg('A system function, such as backup or restore, '+
				'is currently being run by '+FErrorUserName+', please try again later.','Multi-User Problem');
end;

procedure TErrorTable.FileIsLockedMsg;
begin
	ErrorDlg('A system function, such as backup or restore, '+
				'is currently being run by '+FErrorUserName+', please try again later.','Multi-User Problem');
end;

procedure TErrorTable.DirIsLockedMsg;
begin
	ErrorDlg('The directory for the database is locked.','Internal Data Error');
end;

procedure TErrorTable.MultipleNetFilesMsg;
begin
	ErrorDlg('The application was previously not closed properly or '+
				'there could possibly be an improper installation, please '+
				'contact technical support for assistance.','Database Problem');
	DeleteFile(FErrorFileName);
end;

procedure TErrorTable.DetailRecordsExistMsg;
begin
	ErrorDlg('This '+FTableDescription+' cannot be deleted until all dependent data is deleted '+
				'first.','Data Entry Error');
end;

procedure TErrorTable.OtherErrorsMsg(E: Exception);
begin
	ErrorDlg(E.Message,'Unexpected Internal Data Error');
end;

function TableCallBack(ecbType: CBType; ClientData: Longint;
           				  var CBInfo: Pointer): CBRType;
begin
	if (TErrorTable(ClientData).State=dsBrowse) then
		TErrorTable(ClientData).Refresh;
end;

function TErrorTable.ErrorDlg(DisplayMsg: string; TitleMsg: string): Integer;
var
	pTitleMsg: array[0..256] of Char;
	pDisplayMsg: array[0..256] of Char;
begin
	StrPCopy(pTitleMsg,TitleMsg);
	StrPCopy(pDisplayMsg,DisplayMsg);
   Result:=Application.MessageBox(pDisplayMsg,pTitleMsg,
   										 mb_Ok+mb_IconExclamation+mb_DefButton1);
end;

end.
